<?php
require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/includes/functions.php';
require_once __DIR__ . '/api/esim-api.php';

$pageTitle = 'تکمیل خرید';
$error = '';
$isLoggedIn = isLoggedIn();

// Get package and country from URL
$packageCode = $_GET['package'] ?? '';
$countryCode = $_GET['country'] ?? '';

if (empty($packageCode) || empty($countryCode)) {
    header('Location: /');
    exit;
}

// Get package details from API
$api = new eSIMAPI();
$package = $api->getPackageByCode($packageCode);

if (!$package) {
    header('Location: /');
    exit;
}

// Calculate prices
$priceUsd = $package['price'] / 10000;
$priceIrr = usdToIrr($priceUsd);
$volumeGB = round($package['volume'] / 1073741824, 1);
$dataDisplay = $volumeGB . ' GB';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    
    // Logged in user confirming purchase
    if ($isLoggedIn && isset($_POST['confirm_purchase'])) {
        $userId = $_SESSION['user_id'];
        
        $orderData = [
            'country_code' => $countryCode,
            'country_name' => $package['location'],
            'package_name' => $package['name'],
            'package_code' => $packageCode,
            'data_amount' => $dataDisplay,
            'validity_days' => (int)$package['duration']
        ];
        
        $orderId = createOrder($userId, $orderData, $priceUsd);
        
        if ($orderId) {
            header('Location: /proxy-payment.php?order=' . $orderId);
            exit;
        } else {
            $error = 'خطا در ایجاد سفارش. لطفاً دوباره تلاش کنید';
        }
    }
    
    // Guest user login
    elseif (!$isLoggedIn && isset($_POST['login'])) {
        $mobile = sanitize($_POST['mobile'] ?? '');
        $password = $_POST['password'] ?? '';
        
        if (empty($mobile) || empty($password)) {
            $error = 'لطفاً شماره موبایل و رمز عبور را وارد کنید';
        } elseif (!validateMobile($mobile)) {
            $error = 'شماره موبایل نامعتبر است';
        } else {
            if (loginUser($mobile, $password)) {
                $user = getUserByMobile($mobile);
                
                $orderData = [
                    'country_code' => $countryCode,
                    'country_name' => $package['location'],
                    'package_name' => $package['name'],
                    'package_code' => $packageCode,
                    'data_amount' => $dataDisplay,
                    'validity_days' => (int)$package['duration']
                ];
                
                $orderId = createOrder($user['id'], $orderData, $priceUsd);
                
                if ($orderId) {
                    header('Location: /proxy-payment.php?order=' . $orderId);
                    exit;
                } else {
                    $error = 'خطا در ایجاد سفارش';
                }
            } else {
                $error = 'شماره موبایل یا رمز عبور اشتباه است';
            }
        }
    }
    
    // Guest user registration
    elseif (!$isLoggedIn && isset($_POST['register'])) {
        $mobile = sanitize($_POST['reg_mobile'] ?? '');
        $email = sanitize($_POST['reg_email'] ?? '');
        $password = $_POST['reg_password'] ?? '';
        
        if (empty($mobile) || empty($email) || empty($password)) {
            $error = 'لطفاً تمام فیلدها را پر کنید';
        } elseif (!validateMobile($mobile)) {
            $error = 'شماره موبایل نامعتبر است';
        } elseif (!validateEmail($email)) {
            $error = 'آدرس ایمیل نامعتبر است';
        } elseif (strlen($password) < 6) {
            $error = 'رمز عبور باید حداقل 6 کاراکتر باشد';
        } else {
            if (getUserByMobile($mobile)) {
                $error = 'این شماره موبایل قبلاً ثبت شده است';
            } else {
                $userId = createUser($mobile, $email, $password);
                
                if ($userId) {
                    loginUser($mobile, $password);
                    
                    $orderData = [
                        'country_code' => $countryCode,
                        'country_name' => $package['location'],
                        'package_name' => $package['name'],
                        'package_code' => $packageCode,
                        'data_amount' => $dataDisplay,
                        'validity_days' => (int)$package['duration']
                    ];
                    
                    $orderId = createOrder($userId, $orderData, $priceUsd);
                    
                    if ($orderId) {
                        header('Location: /proxy-payment.php?order=' . $orderId);
                        exit;
                    } else {
                        $error = 'خطا در ایجاد سفارش';
                    }
                } else {
                    $error = 'خطا در ثبت نام';
                }
            }
        }
    }
}

include __DIR__ . '/includes/header.php';
?>

<style>
.checkout-container {
    max-width: 900px;
    margin: 0 auto;
}

.checkout-box {
    background: white;
    border-radius: 12px;
    padding: 30px;
    margin-bottom: 25px;
    box-shadow: 0 3px 15px rgba(0, 0, 0, 0.1);
}

.checkout-box h2 {
    color: #667eea;
    margin-bottom: 20px;
    padding-bottom: 15px;
    border-bottom: 2px solid #f0f0f0;
}

.package-summary {
    display: flex;
    gap: 20px;
    align-items: center;
    margin-bottom: 20px;
}

.package-flag-large {
    width: 100px;
    height: 70px;
    object-fit: cover;
    border-radius: 8px;
    box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
}

.package-info {
    flex: 1;
}

.package-info h3 {
    color: #333;
    margin-bottom: 15px;
}

.info-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
    gap: 15px;
}

.info-item {
    background: #f8f9fa;
    padding: 12px;
    border-radius: 6px;
}

.info-item i {
    color: #667eea;
    margin-left: 5px;
}

.info-item strong {
    display: block;
    color: #999;
    font-size: 12px;
    margin-bottom: 5px;
}

.price-box {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    padding: 20px;
    border-radius: 10px;
    text-align: center;
    margin: 20px 0;
}

.price-box .label {
    font-size: 14px;
    opacity: 0.9;
}

.price-box .amount {
    font-size: 32px;
    font-weight: bold;
    margin: 10px 0;
}

.alert {
    padding: 15px;
    border-radius: 8px;
    margin-bottom: 20px;
}

.alert-error {
    background: #fee;
    color: #c33;
    border: 1px solid #fcc;
}

.user-info-box {
    background: #e7f3ff;
    padding: 20px;
    border-radius: 8px;
    margin-bottom: 20px;
}

.user-info-box p {
    margin: 10px 0;
    color: #333;
}

.auth-forms {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 25px;
}

.auth-form-box {
    background: #f8f9fa;
    padding: 25px;
    border-radius: 10px;
}

.auth-form-box h3 {
    color: #667eea;
    margin-bottom: 20px;
    text-align: center;
}

.form-group {
    margin-bottom: 18px;
}

.form-group label {
    display: block;
    margin-bottom: 6px;
    font-weight: 600;
    color: #333;
}

.form-group input {
    width: 100%;
    padding: 12px;
    border: 2px solid #ddd;
    border-radius: 6px;
    font-size: 15px;
    transition: border 0.3s;
}

.form-group input:focus {
    outline: none;
    border-color: #667eea;
}

.form-group small {
    display: block;
    margin-top: 5px;
    color: #666;
    font-size: 12px;
}

.btn {
    width: 100%;
    padding: 14px;
    border: none;
    border-radius: 8px;
    font-size: 16px;
    font-weight: bold;
    cursor: pointer;
    transition: all 0.3s;
}

.btn-primary {
    background: #667eea;
    color: white;
}

.btn-primary:hover {
    background: #5568d3;
}

.btn-success {
    background: #28a745;
    color: white;
}

.btn-success:hover {
    background: #218838;
}

@media (max-width: 768px) {
    .auth-forms {
        grid-template-columns: 1fr;
    }
    
    .package-summary {
        flex-direction: column;
    }
}
</style>

<div class="container">
    <div class="checkout-container">
        <!-- Package Info -->
        <div class="checkout-box">
            <h2><i class="fas fa-shopping-bag"></i> اطلاعات بسته</h2>
            
            <div class="package-summary">
                <img src="https://flagcdn.com/w160/<?php echo strtolower($countryCode); ?>.png" 
                     alt="<?php echo htmlspecialchars($package['location']); ?>"
                     class="package-flag-large"
                     onerror="this.src='https://via.placeholder.com/100x70'">
                
                <div class="package-info">
                    <h3><?php echo htmlspecialchars($package['name']); ?></h3>
                    
                    <div class="info-grid">
                        <div class="info-item">
                            <strong>حجم داده</strong>
                            <i class="fas fa-database"></i>
                            <?php echo $dataDisplay; ?>
                        </div>
                        <div class="info-item">
                            <strong>مدت اعتبار</strong>
                            <i class="fas fa-calendar-alt"></i>
                            <?php echo $package['duration']; ?> روز
                        </div>
                        <div class="info-item">
                            <strong>نوع شبکه</strong>
                            <i class="fas fa-signal"></i>
                            <?php echo $package['speed'] ?? '4G/5G'; ?>
                        </div>
                    </div>
                </div>
            </div>

            <div class="price-box">
                <div class="label">مبلغ قابل پرداخت</div>
                <div class="amount"><?php echo formatPrice($priceIrr); ?> <span style="font-size: 20px;">تومان</span></div>
            </div>
        </div>

        <!-- Error Message -->
        <?php if ($error): ?>
            <div class="alert alert-error">
                <i class="fas fa-exclamation-circle"></i>
                <?php echo $error; ?>
            </div>
        <?php endif; ?>

        <!-- Logged In User -->
        <?php if ($isLoggedIn): ?>
            <div class="checkout-box">
                <h2><i class="fas fa-credit-card"></i> تکمیل خرید</h2>
                
                <div class="user-info-box">
                    <p><i class="fas fa-user"></i> <strong>شماره موبایل:</strong> <?php echo htmlspecialchars($_SESSION['mobile']); ?></p>
                    <p><i class="fas fa-envelope"></i> <strong>ایمیل:</strong> <?php echo htmlspecialchars($_SESSION['email']); ?></p>
                </div>

                <form method="POST">
                    <button type="submit" name="confirm_purchase" class="btn btn-success">
                        <i class="fas fa-check-circle"></i>
                        پرداخت <?php echo formatPrice($priceIrr); ?> تومان
                    </button>
                </form>

                <p style="text-align: center; margin-top: 15px; color: #666; font-size: 14px;">
                    <i class="fas fa-shield-alt"></i>
                    پرداخت از طریق درگاه امن زرین‌پال
                </p>
            </div>
        
        <!-- Guest User -->
        <?php else: ?>
            <div class="checkout-box">
                <h2><i class="fas fa-user-circle"></i> ورود یا ثبت نام</h2>
                <p style="text-align: center; margin-bottom: 25px; color: #666;">
                    برای تکمیل خرید، لطفاً وارد حساب کاربری خود شوید یا ثبت نام کنید
                </p>

                <div class="auth-forms">
                    <!-- Login Form -->
                    <div class="auth-form-box">
                        <h3><i class="fas fa-sign-in-alt"></i> ورود</h3>
                        <form method="POST">
                            <div class="form-group">
                                <label>شماره موبایل</label>
                                <input type="text" name="mobile" required placeholder="09123456789">
                            </div>
                            <div class="form-group">
                                <label>رمز عبور</label>
                                <input type="password" name="password" required placeholder="رمز عبور">
                            </div>
                            <button type="submit" name="login" class="btn btn-primary">
                                ورود و ادامه خرید
                            </button>
                        </form>
                    </div>

                    <!-- Register Form -->
                    <div class="auth-form-box">
                        <h3><i class="fas fa-user-plus"></i> ثبت نام</h3>
                        <form method="POST">
                            <div class="form-group">
                                <label>شماره موبایل</label>
                                <input type="text" name="reg_mobile" required placeholder="09123456789">
                            </div>
                            <div class="form-group">
                                <label>ایمیل</label>
                                <input type="email" name="reg_email" required placeholder="email@example.com">
                                <small>eSIM به این ایمیل ارسال می‌شود</small>
                            </div>
                            <div class="form-group">
                                <label>رمز عبور</label>
                                <input type="password" name="reg_password" required placeholder="حداقل 6 کاراکتر">
                            </div>
                            <button type="submit" name="register" class="btn btn-success">
                                ثبت نام و ادامه خرید
                            </button>
                        </form>
                    </div>
                </div>
            </div>
        <?php endif; ?>
    </div>
</div>

<?php include __DIR__ . '/includes/footer.php'; ?>

